/*! Scan WebAPI Sample Version 1.0.0 Copyright (C) SEIKO EPSON CORPORATION 2022. All rights reserved. */

window.addEventListener(
  "beforeunload",
  function (e) {
    saveIpAddress(getIpAddress());
    const msg = "Do you want to leave this site?";
    e.returnValue = msg;
    return msg;
  },
  false
);

const API_TABLE = [
  {
    name: "Connect",
    id: "connectAPI",
    url: "/api/connect",
    method: "POST",
    authorization: false,
    body: {
      timeout: 300,
    },
  },
  {
    name: "Disconnect",
    id: "disconnectAPI",
    url: "/api/disconnect",
    method: "POST",
    authorization: true,
    body: null,
  },
  {
    name: "ScanSetting",
    id: "scanSettingAPI",
    url: "/api/scan/setting/check",
    method: "POST",
    authorization: true,
    body: {
      images: [
        {
          format: "jpeg",
          type: "grayscale",
        },
      ],
    },
  },
  {
    name: "ScanStart",
    id: "scanStartAPI",
    url: "/api/scan/start/check",
    method: "POST",
    authorization: true,
    body: {
      limit: 1,
    },
  },
  {
    name: "ScanCancel",
    id: "scanCancelAPI",
    url: "/api/scan/cancel",
    method: "POST",
    authorization: true,
    body: null,
  },
  {
    name: "GetDocList",
    id: "getDocListAPI",
    url: "/api/docs",
    method: "GET",
    authorization: true,
    body: null,
  },
  {
    name: "GetImage",
    id: "getImageAPI",
    url: "/api/docs/1/00001_check_front.tif",
    method: "GET",
    authorization: true,
    body: null,
  },
  {
    name: "DeleteAllDocs",
    id: "deleteDocsAPI",
    url: "/api/docs",
    method: "DELETE",
    authorization: true,
    body: null,
  },
  {
    name: "GetDeviceStatus",
    id: "deviceStatusAPI",
    url: "/api/device/status",
    method: "GET",
    authorization: false,
    body: null,
  },
];

function init() {
  setIpAddress(loadIpAddress());
  for (const api of API_TABLE) {
    addAPIList(api.name, api.method, api.id);
  }
  selectAPI("connectAPI");
}

function addAPIList(name, method, id) {
  const listElm = document.getElementById("apiList");

  const aTag = document.createElement("a");
  aTag.id = id;
  aTag.href = "javascript:void(0)";
  aTag.classList.add("list-group-item", "list-group-item-action");
  aTag.setAttribute("onclick", "selectAPI(this.id)");
  listElm.appendChild(aTag);

  const methodSpanTag = document.createElement("span");
  methodSpanTag.classList.add("badge", "method-badge");
  if (method == "GET") {
    methodSpanTag.classList.add("bg-success");
  } else if (method == "POST") {
    methodSpanTag.classList.add("bg-warning");
  } else if (method == "DELETE") {
    methodSpanTag.classList.add("bg-danger");
  } else {
    methodSpanTag.classList.add("bg-info");
  }
  methodSpanTag.textContent = method.length > 4 ? method.substr(0, 3) : method;
  aTag.appendChild(methodSpanTag);

  const nameSpanTag = document.createElement("span");
  nameSpanTag.classList.add("ms-2");
  nameSpanTag.textContent = name;
  aTag.appendChild(nameSpanTag);
}

function setToken(token) {
  document.getElementById("token").value = token;
}

function getToken() {
  return document.getElementById("token").value;
}

function setIpAddress(ip) {
  document.getElementById("ipAddress").value = ip;
}

function getIpAddress() {
  return document.getElementById("ipAddress").value;
}

function selectAPI(id) {
  setSelectedAPIActive(id);

  // Set initial value by selected API
  const info = API_TABLE.find((elm) => elm.id == id);
  // Set method
  document.getElementById("httpMethod").innerText = info.method;
  // Set URL
  document.getElementById("formInputRequestURL").value =
    "https://" + getIpAddress() + info.url;
  // Set headers
  const headerTable = document.getElementById("requestHeadersTableBody");
  headerTable.innerHTML = "";
  if (info.authorization && getToken()) {
    const row = headerTable.insertRow();
    row.insertCell().appendChild(document.createTextNode("Authorization"));
    row.insertCell().appendChild(document.createTextNode(getToken()));
  }
  if (info.body) {
    const row = headerTable.insertRow();
    row.insertCell().appendChild(document.createTextNode("Content-Type"));
    row
      .insertCell()
      .appendChild(document.createTextNode("application/json; charset=utf-8"));
  }
  // Set body
  if (info.body) {
    document.getElementById("requestBody").disabled = false;
    document.getElementById("requestBody").value = prettierJson(info.body);
  } else {
    document.getElementById("requestBody").value = "";
    document.getElementById("requestBody").disabled = true;
  }
}

function selectGetImage(url) {
  const info = API_TABLE.find((elm) => elm.id == "getImageAPI");
  if (info) {
    info.url = url;
    selectAPI("getImageAPI");
  }
}

function validateBodyJson() {
  try {
    const bodyText = document.getElementById("requestBody").value;
    if (bodyText.length > 0) {
      JSON.parse(bodyText);
      document.getElementById("requestBody").classList.remove("invalid-json");
    }
  } catch {
    document.getElementById("requestBody").classList.add("invalid-json");
  }
}

function prettierJson(json) {
  return JSON.stringify(json, null, "  ");
}

function setSelectedAPIActive(id) {
  // Deactivate all API in List
  const elms = document.getElementsByClassName("list-group-item");
  Array.prototype.forEach.call(elms, (elm) => {
    elm.classList.remove("active");
  });
  // Activate selected API
  document.getElementById(id).classList.add("active");
}

function displayResponseArea() {
  document.getElementById("responseArea").classList.remove("d-none");
}

function onFormatButton() {
  const requestBodyElm = document.getElementById("requestBody");
  try {
    if (requestBodyElm.value.length > 0) {
      requestBodyElm.value = prettierJson(JSON.parse(requestBodyElm.value));
    }
  } catch (err) {
    alert("JSON parse error");
    console.log(err);
  }
}

function onSendButton() {
  const method = document.getElementById("httpMethod").innerText;
  const url = document.getElementById("formInputRequestURL").value;
  const body = document.getElementById("requestBody").value;
  const headers = {};
  const headerTable = document.getElementById("requestHeadersTableBody");
  // HTMLCollection is not Array...
  Array.prototype.forEach.call(headerTable.rows, (elm) => {
    headers[elm.cells[0].firstChild.data] = elm.cells[1].firstChild.data;
  });

  const requestOptions = {
    method: method,
    headers: headers,
    body: method == "POST" ? body : null,
    redirect: "follow",
  };

  fetch(url, requestOptions)
    .then((response) => {
      displayResponseArea();
      setResponse(response);
      // Clear token string when disconnect is success
      if (response.status == 200 && url.indexOf("/api/disconnect") != -1) {
        setToken("");
      }
    })
    .catch((err) => {
      console.log("Request timed out");
    });
}

async function setResponse(response) {
  // Display status code
  const statusCodeElm = document.getElementById("statusCode");
  statusCodeElm.classList.remove("bg-success", "bg-danger");
  if (response.status == 200) {
    statusCodeElm.classList.add("bg-success");
  } else {
    statusCodeElm.classList.add("bg-danger");
  }
  statusCodeElm.innerHTML = response.status + " " + response.statusText;

  // Display response body
  const contentType = response.headers.get("Content-Type");
  if (contentType) {
    if (contentType.indexOf("application/json") != -1) {
      // Display as a JSON string
      const body = await response.json();

      // Replace image url with link
      const regexpUrl = /(\/(api)\/(docs)\/([0-9]+)\/([a-zA-Z0-9.\-_]+))/g;
      const makeLink = (all, url, h, href) => {
        return (
          '<a href="#" target="_blank" onclick="selectGetImage(\'' +
          url +
          "'); return false;\">" +
          url +
          "</a>"
        );
      };
      const responseBodyElm = document.getElementById("responseJsonBody");
      responseBodyElm.innerHTML = prettierJson(body)
        .replaceAll("\n", "<br/>")
        .replaceAll(" ", "&nbsp;")
        .replace(regexpUrl, makeLink);
      responseBodyElm.setAttribute("raw-text", JSON.stringify(body));

      switchBodyFormat("responseJsonBody");
      // when connect api response
      if (body["token"]) {
        setToken(body["token"]);
      }
    } else if (
      contentType.indexOf("image/jpeg") != -1 ||
      contentType.indexOf("image/bmp") != -1
    ) {
      // Display JPEG or Bitmap image
      const imageData = await response.blob();
      document
        .getElementById("responseImageBody")
        .setAttribute("src", window.URL.createObjectURL(imageData));
      switchBodyFormat("responseImageBody");
    } else if (contentType.indexOf("image/tiff") != -1) {
      // Display Tiff image with tiff.js
      const imageData = await response.arrayBuffer();
      const tiff = new Tiff({ buffer: imageData });
      document
        .getElementById("responseImageBody")
        .setAttribute("src", tiff.toDataURL());
      switchBodyFormat("responseImageBody");
    } else if (contentType.indexOf("application/zip") != -1) {
      // Download zip file
      const blobFile = await response.blob();
      const link = document.createElement("a");
      link.href = window.URL.createObjectURL(blobFile);
      const getCurrentDateTime = () => {
        const pad = (num, len = 2) => {
          // Zero padding...
          return (Array(len).join("0") + num).slice(-len);
        };
        const date = new Date();
        return (
          date.getFullYear() +
          pad(date.getMonth() + 1) +
          pad(date.getDate()) +
          pad(date.getHours()) +
          pad(date.getMinutes()) +
          pad(date.getSeconds())
        );
      };
      link.download = getCurrentDateTime() + "_log.zip";
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
    } else {
      // not support type..
      console.log("not supported media type");
    }
  }
  // Display response headers
  const headerTable = document.getElementById("responseHeadersTableBody");
  headerTable.innerHTML = "";

  response.headers.forEach((value, key) => {
    const row = headerTable.insertRow();
    row.insertCell().appendChild(document.createTextNode(key));
    row.insertCell().appendChild(document.createTextNode(value));
  });
}

function onResponseTab(id) {
  if (id == "responseBodyTab") {
    document.getElementById("responseBodyTab").classList.add("active");
    document.getElementById("responseHeaderTab").classList.remove("active");
    switchDisplayResponse("responseBody");
  } else if (id == "responseHeaderTab") {
    document.getElementById("responseBodyTab").classList.remove("active");
    document.getElementById("responseHeaderTab").classList.add("active");
    switchDisplayResponse("responseHeaders");
  }
}

function switchDisplayResponse(id) {
  document.getElementById("responseBody").classList.add("d-none");
  document.getElementById("responseHeaders").classList.add("d-none");
  document.getElementById(id).classList.remove("d-none");
}

function switchBodyFormat(id) {
  document.getElementById("responseJsonBody").classList.add("d-none");
  document.getElementById("responseImageBody").classList.add("d-none");
  document.getElementById(id).classList.remove("d-none");
}

function saveIpAddress(ip) {
  if (window.localStorage) {
    window.localStorage.setItem("ipaddress", ip);
  }
}

function loadIpAddress() {
  if (window.localStorage) {
    const ip = window.localStorage.getItem("ipaddress");
    if (ip) {
      return ip;
    }
  }
  // default value
  return "192.168.192.168";
}
