//
//  ePOS_DeviceXMLSampleViewController.m
//  ePOS-DeviceXMLSample
//
//  Copyright (c) 2013 SEIKO EPSON. All rights reserved.
//

#import "ePOS_DeviceXMLSampleViewController.h"

@interface ePOS_DeviceXMLSampleViewController ()

@property (nonatomic) NSInputStream *inputStream;
@property (nonatomic) NSOutputStream *outputStream;
@property (copy, nonatomic) NSString *printerId;
@property (copy, nonatomic) NSString *scannerId;
@property (copy, nonatomic) NSString *clientId;
@property (copy, nonatomic) NSString *dataId;
@property (copy, nonatomic) NSString *tempData;
@property BOOL isPrinterOpened;
@property BOOL isConnected;
@property BOOL isReconnecting;

@end

@implementation ePOS_DeviceXMLSampleViewController

@synthesize inputStream = _inputStream;
@synthesize outputStream = _outputStream;
@synthesize printerId = _printerId;
@synthesize scannerId = _scannerId;
@synthesize clientId = _clientId;
@synthesize dataId = _dataId;
@synthesize tempData = _tempData;
@synthesize isPrinterOpened = _isPrinterOpened;
@synthesize isConnected = _isConnected;
@synthesize isReconnecting = _isReconnecting;

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    UIApplication *app = [UIApplication sharedApplication];
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(initialize:)
                                                 name:UIApplicationDidEnterBackgroundNotification
                                               object:app];
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(initialize:)
                                                 name:UIApplicationWillTerminateNotification
                                               object:app];
    
	// Do any additional setup after loading the view, typically from a nib.
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

// Connect or disconnect to server.
- (IBAction)connectOrDisconnectClicked:(id)sender
{
    self.buttonConnectOrDisconnect.enabled = NO;
    
    // Disconnect if already connected to server.
    if (self.isConnected) {
        [self disconnect];
        [self closeSocket];
        [self appendConsole:@"Disconnected."];
    }
    
    // Cancel reconnecting process.
    else if (self.isReconnecting) {
        self.isReconnecting = NO;
        [self closeSocket];
        [self appendConsole:@"Canceled reconnection."];
    }
    
    // Connect to server.
    else {
        [self connect];
    }
}

// Send print request to server.
- (IBAction)printClicked:(id)sender
{
    if (!self.isPrinterOpened) {
        [self appendConsole:@"Printer is not opened."];
        return;
    }
    
    NSString *text = self.textViewOnData.text;
    NSString *req = [[NSString alloc] initWithFormat:@"<device_data><sequence>100</sequence><device_id>%@</device_id><data><type>print</type><timeout>10000</timeout><printdata><epos-print xmlns=\"http://www.epson-pos.com/schemas/2011/03/epos-print\"><text lang='ja' smooth='true'>Sample Print&#10;%@</text><cut type=\"feed\" /></epos-print></printdata></data></device_data>\0", self.printerId, text];
    const uint8_t *rawstring = (const uint8_t *)[req UTF8String];
    
    [self.outputStream write:rawstring maxLength:[req lengthOfBytesUsingEncoding:NSUTF8StringEncoding]];
}

// An NSStream delegate callback that's called when events happen on our network stream.
- (void)stream:(NSStream *)stream handleEvent:(NSStreamEvent)eventCode
{
    switch(eventCode) {
            
        case NSStreamEventHasBytesAvailable:
            if (stream == self.inputStream) {
                @try {
                    uint8_t buf[1024];
                    unsigned int len = 0;
                    NSMutableString *data = [NSMutableString stringWithCapacity:0];
                    
                    // Append temp data.
                    if (self.tempData != nil) {
                        [data appendString:self.tempData];
                        self.tempData = nil;
                    }
                    
                    while ([self.inputStream hasBytesAvailable]) {
                        len = [(NSInputStream *)stream read:buf maxLength:sizeof(buf)];
                        
                        if (len > 0) {
                            [data appendString:[[NSString alloc] initWithBytes:buf length:len encoding:NSUTF8StringEncoding]];
                        }
                    }
                    
                    // Save data temporarily and use next available event if last char is not NULL char.
                    if ([data characterAtIndex:[data length] - 1] != 0) {
                        self.tempData = [NSString stringWithString:data];
                        return;                    }
                    
                    NSArray *tokens = [data componentsSeparatedByString:@"\0"];
                    NSString *str;
                    
                    for (str in tokens) {
                        if ([str isEqualToString:@""]) {
                            continue;
                        }
                        NSString *rootName = [self getRootName:str];
                        
                        // Response of connect request.
                        if ([rootName isEqualToString:@"<connect>"]) {
                            
                            self.isConnected = YES;
                            [self appendConsole:@"Connect to server success."];
                            
                            NSString *currentClientId = [self getStringValue:str nodeName:@"<client_id>"];
                            
                            // Send reconnect message.
                            if (self.isReconnecting) {
                                NSString *req = [[NSString alloc] initWithFormat:@"<reconnect><data><new_client_id>%@</new_client_id><old_client_id>%@</old_client_id><received_id>%@</received_id></data></reconnect>\0", currentClientId, self.clientId, self.dataId];
                                
                                const uint8_t *rawstring = (const uint8_t *)[req UTF8String];
                                
                                [self.outputStream write:rawstring maxLength:[req lengthOfBytesUsingEncoding:NSUTF8StringEncoding]];
                                
                                self.isReconnecting = NO;
                            }
                            else {
                                // Disconnect old connection.
                                if (self.clientId != nil) {
                                    [self disconnect];
                                }
                                
                                [self.buttonConnectOrDisconnect setTitle:@"disconnect" forState:UIControlStateNormal];
                                
                                [self openPrinter];
                                [self openScanner];
                            }
                            
                            self.buttonConnectOrDisconnect.enabled = YES;
                            
                            // Save current clientID.
                            self.clientId = currentClientId;
                            
                        }
                        
                        // Response of open_device request.
                        else if ([rootName isEqualToString:@"<open_device>"]) {
                            NSString *deviceId = [self getStringValue:str nodeName:@"<device_id>"];
                            NSString *code = [self getStringValue:str nodeName:@"<code>"];
                            
                            [self appendConsole:[NSString stringWithFormat:@"%@ : %@", deviceId, code]];
                            
                            if ([deviceId isEqualToString:self.printerId] && [code isEqualToString:@"OK"]) {
                                self.isPrinterOpened = YES;
                            }
                        }
                        
                        // Server and client exchange the data using device_data message.
                        else if ([rootName isEqualToString:@"<device_data>"]) {
                            NSString *type = [self getStringValue:str nodeName:@"<type>"];
                            
                            // Input data from scanner.
                            if ([type isEqualToString:@"ondata"]) {
                                self.textViewOnData.text = [self.textViewOnData.text stringByAppendingFormat:@"%@\n", [self getStringValue:str nodeName:@"<input>"]];
                            }
                            
                            // Response of print request.
                            else if ([type isEqualToString:@"onxmlresult"]) {
                                NSString *result = [self getStringValue:str nodeName:@"<result"];
                                NSRange range = [result rangeOfString:@"true"];
                                
                                if (range.location != NSNotFound) {
                                    [self appendConsole:@"Print successed."];
                                }
                                else {
                                    [self appendConsole:@"Print failed."];
                                }
                            }
                        }
                        
                        // Response of reconnect request.
                        else if ([rootName isEqualToString:@"<reconnect>"]) {
                            if ([[self getStringValue:str nodeName:@"<code>"] isEqualToString:@"OK"]) {
                                self.isPrinterOpened = YES;
                                [self appendConsole:@"Reconnected."];
                            }
                        }
                        
                        // Save latest number of data_id.
                        NSString *latestDataId = [self getStringValue:str nodeName:@"<data_id>"];
                        if (latestDataId != nil) {
                            self.dataId = latestDataId;
                        }
                    }
                }
                @catch (NSException *exception) {
                    NSLog(@"%@", [exception reason]);
                }
            }
            break;
        case NSStreamEventErrorOccurred:
            
            // Continue reconnecting.
            if (self.isReconnecting) {
                [self reconnect];
            }
            
            // Try to reconnect to server when disconncted by error.
            else if (self.isConnected) {
                [self appendConsole:@"Disconnected."];
                [self reconnect];
            }
            
            // Arrive here when failed first connection or canceled reconnection.
            else {
                [self appendConsole:@"Connect to server failed."];
                [self closeSocket];
            }
            
            break;
        case NSStreamEventHasSpaceAvailable:
            break;
        case NSStreamEventNone:
            break;
        case NSStreamEventOpenCompleted:
            break;
        case NSStreamEventEndEncountered:
            break;
    }
}

// Connect to ePOS-Device XML server using tcp socket.
- (void)connect
{
    NSString *urlStr = self.textFieldIpAddress.text;
    
    if (![urlStr isEqualToString:@""]) {
        NSURL *hostAddress = [NSURL URLWithString:urlStr];
        if (!hostAddress) {
            [self appendConsole:[NSString stringWithFormat:@"%@ is not a valid host.", urlStr]];
            self.buttonConnectOrDisconnect.enabled = YES;
            return;
        }
        
        CFReadStreamRef readStream;
        CFWriteStreamRef writeStream;
        CFStreamCreatePairWithSocketToHost(NULL, (__bridge CFStringRef)urlStr, 8009, &readStream, &writeStream);
        
        self.inputStream = (__bridge_transfer NSInputStream *)readStream;
        self.outputStream = (__bridge_transfer NSOutputStream *)writeStream;
        [self.inputStream setDelegate:self];
        [self.outputStream setDelegate:self];
        [self.inputStream scheduleInRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
        [self.outputStream scheduleInRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
        [self.inputStream open];
        [self.outputStream open];
    }
}

// Send disconnect message to server.
- (void)disconnect
{
    NSString *req = [[NSString alloc] initWithFormat:@"<disconnect><data><client_id>%@</client_id></data></disconnect>\0", self.clientId];
    
    const uint8_t *rawstring = (const uint8_t *)[req UTF8String];
    
    [self.outputStream write:rawstring maxLength:[req lengthOfBytesUsingEncoding:NSUTF8StringEncoding]];
    
    self.clientId = nil;
}

// Reconnect to server.
- (void)reconnect
{
    self.isReconnecting = YES;
    
    [self appendConsole:@"Reconnecting."];
    
    [self closeSocket];
    
    // Call connect method after delay.
    [self performSelector:@selector(connect) withObject:nil afterDelay:5.0];
}

// Close and initialize input and output stream.
- (void)closeSocket
{
    if (self.inputStream != nil) {
        self.inputStream.delegate = nil;
        [self.inputStream removeFromRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
        [self.inputStream close];
        self.inputStream = nil;
    }
    if (self.outputStream != nil) {
        self.outputStream.delegate = nil;
        [self.outputStream removeFromRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
        [self.outputStream close];
        self.outputStream = nil;
    }
    
    self.isConnected = NO;
    self.isPrinterOpened = NO;
    
    if (!self.isReconnecting) {
        [self.buttonConnectOrDisconnect setTitle:@"connect" forState:UIControlStateNormal];
    }
    self.buttonConnectOrDisconnect.enabled = YES;
}

// Send open_device message to use printer.
- (void)openPrinter
{
    self.printerId = self.textFieldPrinterId.text;
    NSString *req = [[NSString alloc] initWithFormat:@"<open_device><device_id>%@</device_id><data><type>type_printer</type></data></open_device>\0", self.printerId];
    
    const uint8_t *rawstring = (const uint8_t *)[req UTF8String];
    
    [self.outputStream write:rawstring maxLength:[req lengthOfBytesUsingEncoding:NSUTF8StringEncoding]];
}

// Send open_device message to use scanner.
- (void)openScanner
{
    self.scannerId = self.textFieldScannerId.text;
    NSString *req = [[NSString alloc] initWithFormat:@"<open_device><device_id>%@</device_id><data><type>type_scanner</type><buffer>true</buffer></data></open_device>\0", self.scannerId];
    
    const uint8_t *rawstring = (const uint8_t *)[req UTF8String];
    
    [self.outputStream write:rawstring maxLength:[req lengthOfBytesUsingEncoding:NSUTF8StringEncoding]];
}

// Get name of root element from XML string.
- (NSString *)getRootName:(NSString *)xmlString
{
    NSRange range = [xmlString rangeOfString:@"<.+?>" options:NSRegularExpressionSearch];
    
    return range.length ? [xmlString substringWithRange:range] : nil;
}

// Get value of string node.
- (NSString *)getStringValue:(NSString *)xmlString nodeName:(NSString *)name
{
    NSRange range = [xmlString rangeOfString:name];
    
    NSMutableString *endStr = [[NSMutableString alloc] initWithString:name];
    [endStr insertString:@"/" atIndex:1];
    
    NSRange end = [xmlString rangeOfString:endStr];
    int begin = range.location + range.length;
    NSRange subLength = NSMakeRange(begin, end.location - begin);
    
    return subLength.length ? [xmlString substringWithRange:subLength] : nil;
}

// Append text to textview for console.
- (void)appendConsole:(NSString *)str
{
    NSString *text = [str stringByAppendingString:@"\n"];
    text = [text stringByAppendingString:self.textViewConsole.text];
    self.textViewConsole.text = text;
}

// Initialize connection state.
- (void)initialize:(NSNotification *)notification
{
    if (self.isConnected) {
        [self disconnect];
        [self closeSocket];
        [self appendConsole:@"Disconnected."];
    }
    else if (self.isReconnecting) {
        self.isReconnecting = NO;
        [self closeSocket];
        [self appendConsole:@"Canceled reconnection."];
    }
}

// Delegate method to close software keyboard when return key pushed.
- (BOOL)textFieldShouldReturn:(UITextField *)textField
{
    [textField resignFirstResponder];
    return YES;
}

@end
